/* --------------------------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
'use strict';
var code = require('vscode');
var proto = require('./protocol');
var is = require('./utils/is');
var protocolCompletionItem_1 = require('./protocolCompletionItem');
var protocolCodeLens_1 = require('./protocolCodeLens');
function asDiagnostics(diagnostics) {
    return diagnostics.map(asDiagnostic);
}
exports.asDiagnostics = asDiagnostics;
function asDiagnostic(diagnostic) {
    var result = new code.Diagnostic(asRange(diagnostic.range), diagnostic.message, asDiagnosticSeverity(diagnostic.severity));
    if (is.defined(diagnostic.code)) {
        result.code = diagnostic.code;
    }
    if (is.defined(diagnostic.source)) {
        result.source = diagnostic.source;
    }
    return result;
}
exports.asDiagnostic = asDiagnostic;
function asRange(value) {
    if (is.undefined(value)) {
        return undefined;
    }
    else if (is.nil(value)) {
        return null;
    }
    return new code.Range(asPosition(value.start), asPosition(value.end));
}
exports.asRange = asRange;
function asPosition(value) {
    if (is.undefined(value)) {
        return undefined;
    }
    else if (is.nil(value)) {
        return null;
    }
    return new code.Position(value.line, value.character);
}
exports.asPosition = asPosition;
function asDiagnosticSeverity(value) {
    if (is.undefined(value) || is.nil(value)) {
        return code.DiagnosticSeverity.Error;
    }
    switch (value) {
        case proto.DiagnosticSeverity.Error:
            return code.DiagnosticSeverity.Error;
        case proto.DiagnosticSeverity.Warning:
            return code.DiagnosticSeverity.Warning;
        case proto.DiagnosticSeverity.Information:
            return code.DiagnosticSeverity.Information;
        case proto.DiagnosticSeverity.Hint:
            return code.DiagnosticSeverity.Hint;
    }
    return code.DiagnosticSeverity.Error;
}
exports.asDiagnosticSeverity = asDiagnosticSeverity;
function asHover(hover) {
    if (is.undefined(hover)) {
        return undefined;
    }
    if (is.nil(hover)) {
        return null;
    }
    return new code.Hover(hover.contents, is.defined(hover.range) ? asRange(hover.range) : undefined);
}
exports.asHover = asHover;
function asCompletionResult(result) {
    if (Array.isArray(result)) {
        var items = result;
        return items.map(asCompletionItem);
    }
    var list = result;
    return new code.CompletionList(list.items.map(asCompletionItem), list.isIncomplete);
}
exports.asCompletionResult = asCompletionResult;
function set(value, func) {
    if (is.defined(value)) {
        func();
    }
}
function asCompletionItem(item) {
    var result = new protocolCompletionItem_1.default(item.label);
    set(item.detail, function () { return result.detail = item.detail; });
    set(item.documentation, function () { return result.documentation = item.documentation; });
    set(item.filterText, function () { return result.filterText = item.filterText; });
    set(item.insertText, function () { return result.insertText = item.insertText; });
    // Protocol item kind is 1 based, codes item kind is zero based.
    set(item.kind, function () { return result.kind = item.kind - 1; });
    set(item.sortText, function () { return result.sortText = item.sortText; });
    set(item.textEdit, function () { return result.textEdit = asTextEdit(item.textEdit); });
    set(item.data, function () { return result.data = item.data; });
    return result;
}
exports.asCompletionItem = asCompletionItem;
function asTextEdit(edit) {
    return new code.TextEdit(asRange(edit.range), edit.newText);
}
exports.asTextEdit = asTextEdit;
function asTextEdits(items) {
    return items.map(asTextEdit);
}
exports.asTextEdits = asTextEdits;
function asSignatureHelp(item) {
    var result = new code.SignatureHelp();
    set(item.activeParameter, function () { return result.activeParameter = item.activeParameter; });
    set(item.activeSignature, function () { return result.activeSignature = item.activeSignature; });
    set(item.signatures, function () { return result.signatures = asSignatureInformations(item.signatures); });
    return result;
}
exports.asSignatureHelp = asSignatureHelp;
function asSignatureInformations(items) {
    return items.map(asSignatureInformation);
}
exports.asSignatureInformations = asSignatureInformations;
function asSignatureInformation(item) {
    var result = new code.SignatureInformation(item.label);
    set(item.documentation, function () { return result.documentation = item.documentation; });
    set(item.parameters, function () { return result.parameters = asParameterInformations(item.parameters); });
    return result;
}
exports.asSignatureInformation = asSignatureInformation;
function asParameterInformations(item) {
    return item.map(asParameterInformation);
}
exports.asParameterInformations = asParameterInformations;
function asParameterInformation(item) {
    var result = new code.ParameterInformation(item.label);
    set(item.documentation, function () { return result.documentation = item.documentation; });
    return result;
}
exports.asParameterInformation = asParameterInformation;
function asDefinitionResult(item) {
    if (is.array(item)) {
        return item.map(asLocation);
    }
    else {
        return asLocation(item);
    }
}
exports.asDefinitionResult = asDefinitionResult;
function asLocation(item) {
    if (is.undefined(item)) {
        return undefined;
    }
    if (is.nil(item)) {
        return null;
    }
    return new code.Location(code.Uri.parse(item.uri), asRange(item.range));
}
exports.asLocation = asLocation;
function asReferences(values) {
    return values.map(asLocation);
}
exports.asReferences = asReferences;
function asDocumentHighlights(values) {
    return values.map(asDocumentHighlight);
}
exports.asDocumentHighlights = asDocumentHighlights;
function asDocumentHighlight(item) {
    var result = new code.DocumentHighlight(asRange(item.range));
    set(item.kind, function () { return result.kind = asDocumentHighlightKind(item.kind); });
    return result;
}
exports.asDocumentHighlight = asDocumentHighlight;
function asDocumentHighlightKind(item) {
    switch (item) {
        case proto.DocumentHighlightKind.Text:
            return code.DocumentHighlightKind.Text;
        case proto.DocumentHighlightKind.Read:
            return code.DocumentHighlightKind.Read;
        case proto.DocumentHighlightKind.Write:
            return code.DocumentHighlightKind.Write;
    }
    return code.DocumentHighlightKind.Text;
}
exports.asDocumentHighlightKind = asDocumentHighlightKind;
function asSymbolInformations(values, uri) {
    return values.map(function (information) { return asSymbolInformation(information, uri); });
}
exports.asSymbolInformations = asSymbolInformations;
function asSymbolInformation(item, uri) {
    // Symbol kind is one based in the protocol and zero based in code.
    var result = new code.SymbolInformation(item.name, item.kind - 1, asRange(item.location.range), item.location.uri ? code.Uri.parse(item.location.uri) : uri);
    set(item.containerName, function () { return result.containerName = item.containerName; });
    return result;
}
exports.asSymbolInformation = asSymbolInformation;
function asCommand(item) {
    var result = { title: item.title, command: item.command };
    set(item.arguments, function () { return result.arguments = item.arguments; });
    return result;
}
exports.asCommand = asCommand;
function asCommands(items) {
    return items.map(asCommand);
}
exports.asCommands = asCommands;
function asCodeLens(item) {
    var result = new protocolCodeLens_1.default(asRange(item.range));
    if (is.defined(item.command))
        result.command = asCommand(item.command);
    if (is.defined(item.data))
        result.data = item.data;
    return result;
}
exports.asCodeLens = asCodeLens;
function asCodeLenses(items) {
    return items.map(asCodeLens);
}
exports.asCodeLenses = asCodeLenses;
function asWorkspaceEdit(item) {
    var result = new code.WorkspaceEdit();
    var keys = Object.keys(item.changes);
    keys.forEach(function (key) { return result.set(code.Uri.parse(key), asTextEdits(item.changes[key])); });
    return result;
}
exports.asWorkspaceEdit = asWorkspaceEdit;
