var AutoCollectConsole = require("./AutoCollection/Console");
var AutoCollectExceptions = require("./AutoCollection/Exceptions");
var AutoCollectPerformance = require("./AutoCollection/Performance");
var AutoCollectRequests = require("./AutoCollection/Requests");
var Client = require("./Library/Client");
var Logging = require("./Library/Logging");
/**
 * The singleton meta class for the default client of the client. This class is used to setup/start and configure
 * the auto-collection behavior of the application insights module.
 */
var ApplicationInsights = (function () {
    function ApplicationInsights() {
    }
    /**
     * Initializes a client with the given instrumentation key, if this is not specified, the value will be
     * read from the environment variable APPINSIGHTS_INSTRUMENTATIONKEY
     * @returns {ApplicationInsights/Client} a new client
     */
    ApplicationInsights.getClient = function (instrumentationKey) {
        return new Client(instrumentationKey);
    };
    /**
     * Initializes the default client of the client and sets the default configuration
     * @param instrumentationKey the instrumentation key to use. Optional, if this is not specified, the value will be
     * read from the environment variable APPINSIGHTS_INSTRUMENTATIONKEY
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setup = function (instrumentationKey) {
        if (!ApplicationInsights.client) {
            ApplicationInsights.client = ApplicationInsights.getClient(instrumentationKey);
            ApplicationInsights._console = new AutoCollectConsole(ApplicationInsights.client);
            ApplicationInsights._exceptions = new AutoCollectExceptions(ApplicationInsights.client);
            ApplicationInsights._performance = new AutoCollectPerformance(ApplicationInsights.client);
            ApplicationInsights._requests = new AutoCollectRequests(ApplicationInsights.client);
        }
        else {
            Logging.info("The default client is already setup");
        }
        if (ApplicationInsights.client && ApplicationInsights.client.channel) {
            ApplicationInsights.client.channel.setOfflineMode(ApplicationInsights._isOfflineMode);
        }
        return ApplicationInsights;
    };
    /**
     * Starts automatic collection of telemetry. Prior to calling start no telemetry will be collected
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.start = function () {
        if (!!this.client) {
            ApplicationInsights._isStarted = true;
            ApplicationInsights._console.enable(ApplicationInsights._isConsole);
            ApplicationInsights._exceptions.enable(ApplicationInsights._isExceptions);
            ApplicationInsights._performance.enable(ApplicationInsights._isPerformance);
            ApplicationInsights._requests.enable(ApplicationInsights._isRequests);
        }
        else {
            Logging.warn("Start cannot be called before setup");
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of console tracking (enabled by default)
     * @param value if true console activity will be sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectConsole = function (value) {
        ApplicationInsights._isConsole = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._console.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of exception tracking (enabled by default)
     * @param value if true uncaught exceptions will be sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectExceptions = function (value) {
        ApplicationInsights._isExceptions = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._exceptions.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of performance tracking (enabled by default)
     * @param value if true performance counters will be collected every second and sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectPerformance = function (value) {
        ApplicationInsights._isPerformance = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._performance.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of request tracking (enabled by default)
     * @param value if true requests will be sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectRequests = function (value) {
        ApplicationInsights._isRequests = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._requests.enable(value);
        }
        return ApplicationInsights;
    };
    /**
    * Enable or disable offline mode to cache events when client is offline (disabled by default)
    * @param value if true events that occured while client is offline will be cahced on disk
    * @returns {ApplicationInsights} this class
    */
    ApplicationInsights.setOfflineMode = function (value) {
        ApplicationInsights._isOfflineMode = value;
        if (ApplicationInsights.client && ApplicationInsights.client.channel) {
            ApplicationInsights.client.channel.setOfflineMode(value);
        }
        return ApplicationInsights;
    };
    /**
     * Enables verbose debug logging
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.enableVerboseLogging = function () {
        Logging.enableDebug = true;
        return ApplicationInsights;
    };
    ApplicationInsights._isConsole = true;
    ApplicationInsights._isExceptions = true;
    ApplicationInsights._isPerformance = true;
    ApplicationInsights._isRequests = true;
    ApplicationInsights._isOfflineMode = false;
    ApplicationInsights._isStarted = false;
    return ApplicationInsights;
})();
module.exports = ApplicationInsights;
