/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
"use strict";
var __extends = (this && this.__extends) || function (d, b) {
    for (var p in b) if (b.hasOwnProperty(p)) d[p] = b[p];
    function __() { this.constructor = d; }
    d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
};
var protocol_1 = require('./protocol');
var messages_1 = require('./messages');
var Net = require('net');
var Url = require('url');
var Source = (function () {
    function Source(name, path, id, origin, data) {
        if (id === void 0) { id = 0; }
        this.name = name;
        this.path = path;
        this.sourceReference = id;
        if (origin) {
            this.origin = origin;
        }
        if (data) {
            this.adapterData = data;
        }
    }
    return Source;
}());
exports.Source = Source;
var Scope = (function () {
    function Scope(name, reference, expensive) {
        if (expensive === void 0) { expensive = false; }
        this.name = name;
        this.variablesReference = reference;
        this.expensive = expensive;
    }
    return Scope;
}());
exports.Scope = Scope;
var StackFrame = (function () {
    function StackFrame(i, nm, src, ln, col) {
        if (ln === void 0) { ln = 0; }
        if (col === void 0) { col = 0; }
        this.id = i;
        this.source = src;
        this.line = ln;
        this.column = col;
        this.name = nm;
    }
    return StackFrame;
}());
exports.StackFrame = StackFrame;
var Thread = (function () {
    function Thread(id, name) {
        this.id = id;
        if (name) {
            this.name = name;
        }
        else {
            this.name = 'Thread #' + id;
        }
    }
    return Thread;
}());
exports.Thread = Thread;
var Variable = (function () {
    function Variable(name, value, ref, indexedVariables, namedVariables) {
        if (ref === void 0) { ref = 0; }
        this.name = name;
        this.value = value;
        this.variablesReference = ref;
        if (typeof namedVariables === 'number') {
            this.namedVariables = namedVariables;
        }
        if (typeof indexedVariables === 'number') {
            this.indexedVariables = indexedVariables;
        }
    }
    return Variable;
}());
exports.Variable = Variable;
var Breakpoint = (function () {
    function Breakpoint(verified, line, column, source) {
        this.verified = verified;
        var e = this;
        if (typeof line === 'number') {
            e.line = line;
        }
        if (typeof column === 'number') {
            e.column = column;
        }
        if (source) {
            e.source = source;
        }
    }
    return Breakpoint;
}());
exports.Breakpoint = Breakpoint;
var Module = (function () {
    function Module(id, name) {
        this.id = id;
        this.name = name;
    }
    return Module;
}());
exports.Module = Module;
var CompletionItem = (function () {
    function CompletionItem(label, start, length) {
        if (length === void 0) { length = 0; }
        this.label = label;
        this.start = start;
        this.length = length;
    }
    return CompletionItem;
}());
exports.CompletionItem = CompletionItem;
var StoppedEvent = (function (_super) {
    __extends(StoppedEvent, _super);
    function StoppedEvent(reason, threadId, exception_text) {
        if (exception_text === void 0) { exception_text = null; }
        _super.call(this, 'stopped');
        this.body = {
            reason: reason,
            threadId: threadId
        };
        if (exception_text) {
            var e = this;
            e.body.text = exception_text;
        }
    }
    return StoppedEvent;
}(messages_1.Event));
exports.StoppedEvent = StoppedEvent;
var ContinuedEvent = (function (_super) {
    __extends(ContinuedEvent, _super);
    function ContinuedEvent(threadId, allThreadsContinued) {
        _super.call(this, 'continued');
        this.body = {
            threadId: threadId
        };
        if (typeof allThreadsContinued === 'boolean') {
            this.body.allThreadsContinued = allThreadsContinued;
        }
    }
    return ContinuedEvent;
}(messages_1.Event));
exports.ContinuedEvent = ContinuedEvent;
var InitializedEvent = (function (_super) {
    __extends(InitializedEvent, _super);
    function InitializedEvent() {
        _super.call(this, 'initialized');
    }
    return InitializedEvent;
}(messages_1.Event));
exports.InitializedEvent = InitializedEvent;
var TerminatedEvent = (function (_super) {
    __extends(TerminatedEvent, _super);
    function TerminatedEvent(restart) {
        _super.call(this, 'terminated');
        if (typeof restart === 'boolean') {
            var e = this;
            e.body = {
                restart: restart
            };
        }
    }
    return TerminatedEvent;
}(messages_1.Event));
exports.TerminatedEvent = TerminatedEvent;
var OutputEvent = (function (_super) {
    __extends(OutputEvent, _super);
    function OutputEvent(output, category) {
        if (category === void 0) { category = 'console'; }
        _super.call(this, 'output');
        this.body = {
            category: category,
            output: output
        };
    }
    return OutputEvent;
}(messages_1.Event));
exports.OutputEvent = OutputEvent;
var ThreadEvent = (function (_super) {
    __extends(ThreadEvent, _super);
    function ThreadEvent(reason, threadId) {
        _super.call(this, 'thread');
        this.body = {
            reason: reason,
            threadId: threadId
        };
    }
    return ThreadEvent;
}(messages_1.Event));
exports.ThreadEvent = ThreadEvent;
var BreakpointEvent = (function (_super) {
    __extends(BreakpointEvent, _super);
    function BreakpointEvent(reason, breakpoint) {
        _super.call(this, 'breakpoint');
        this.body = {
            reason: reason,
            breakpoint: breakpoint
        };
    }
    return BreakpointEvent;
}(messages_1.Event));
exports.BreakpointEvent = BreakpointEvent;
var ModuleEvent = (function (_super) {
    __extends(ModuleEvent, _super);
    function ModuleEvent(reason, module) {
        _super.call(this, 'module');
        this.body = {
            reason: reason,
            module: module
        };
    }
    return ModuleEvent;
}(messages_1.Event));
exports.ModuleEvent = ModuleEvent;
(function (ErrorDestination) {
    ErrorDestination[ErrorDestination["User"] = 1] = "User";
    ErrorDestination[ErrorDestination["Telemetry"] = 2] = "Telemetry";
})(exports.ErrorDestination || (exports.ErrorDestination = {}));
var ErrorDestination = exports.ErrorDestination;
;
var DebugSession = (function (_super) {
    __extends(DebugSession, _super);
    function DebugSession(obsolete_debuggerLinesAndColumnsStartAt1, obsolete_isServer) {
        var _this = this;
        _super.call(this);
        var linesAndColumnsStartAt1 = typeof obsolete_debuggerLinesAndColumnsStartAt1 === 'boolean' ? obsolete_debuggerLinesAndColumnsStartAt1 : false;
        this._debuggerLinesStartAt1 = linesAndColumnsStartAt1;
        this._debuggerColumnsStartAt1 = linesAndColumnsStartAt1;
        this._debuggerPathsAreURIs = false;
        this._clientLinesStartAt1 = true;
        this._clientColumnsStartAt1 = true;
        this._clientPathsAreURIs = false;
        this._isServer = typeof obsolete_isServer === 'boolean' ? obsolete_isServer : false;
        this.on('close', function () {
            _this.shutdown();
        });
        this.on('error', function (error) {
            _this.shutdown();
        });
    }
    DebugSession.prototype.setDebuggerPathFormat = function (format) {
        this._debuggerPathsAreURIs = format !== 'path';
    };
    DebugSession.prototype.setDebuggerLinesStartAt1 = function (enable) {
        this._debuggerLinesStartAt1 = enable;
    };
    DebugSession.prototype.setDebuggerColumnsStartAt1 = function (enable) {
        this._debuggerColumnsStartAt1 = enable;
    };
    DebugSession.prototype.setRunAsServer = function (enable) {
        this._isServer = enable;
    };
    /**
     * A virtual constructor...
     */
    DebugSession.run = function (debugSession) {
        // parse arguments
        var port = 0;
        var args = process.argv.slice(2);
        args.forEach(function (val, index, array) {
            var portMatch = /^--server=(\d{4,5})$/.exec(val);
            if (portMatch) {
                port = parseInt(portMatch[1], 10);
            }
        });
        if (port > 0) {
            // start as a server
            console.error("waiting for debug protocol on port " + port);
            Net.createServer(function (socket) {
                console.error('>> accepted connection from client');
                socket.on('end', function () {
                    console.error('>> client connection closed\n');
                });
                var session = new debugSession(false, true);
                session.setRunAsServer(true);
                session.start(socket, socket);
            }).listen(port);
        }
        else {
            // start a session
            //console.error('waiting for debug protocol on stdin/stdout');
            var session_1 = new debugSession(false);
            process.on('SIGTERM', function () {
                session_1.shutdown();
            });
            session_1.start(process.stdin, process.stdout);
        }
    };
    DebugSession.prototype.shutdown = function () {
        if (this._isServer) {
            console.error('shutdown ignored in server mode');
        }
        else {
            // wait a bit before shutting down
            setTimeout(function () {
                process.exit(0);
            }, 100);
        }
    };
    DebugSession.prototype.sendErrorResponse = function (response, codeOrMessage, format, variables, dest) {
        if (dest === void 0) { dest = ErrorDestination.User; }
        var msg;
        if (typeof codeOrMessage === 'number') {
            msg = {
                id: codeOrMessage,
                format: format
            };
            if (variables) {
                msg.variables = variables;
            }
            if (dest & ErrorDestination.User) {
                msg.showUser = true;
            }
            if (dest & ErrorDestination.Telemetry) {
                msg.sendTelemetry = true;
            }
        }
        else {
            msg = codeOrMessage;
        }
        response.success = false;
        response.message = DebugSession.formatPII(msg.format, true, msg.variables);
        if (!response.body) {
            response.body = {};
        }
        response.body.error = msg;
        this.sendResponse(response);
    };
    DebugSession.prototype.runInTerminalRequest = function (args, timeout, cb) {
        this.sendRequest('runInTerminal', args, timeout, cb);
    };
    DebugSession.prototype.dispatchRequest = function (request) {
        var response = new messages_1.Response(request);
        try {
            if (request.command === 'initialize') {
                var args = request.arguments;
                if (typeof args.linesStartAt1 === 'boolean') {
                    this._clientLinesStartAt1 = args.linesStartAt1;
                }
                if (typeof args.columnsStartAt1 === 'boolean') {
                    this._clientColumnsStartAt1 = args.columnsStartAt1;
                }
                if (args.pathFormat !== 'path') {
                    this.sendErrorResponse(response, 2018, 'debug adapter only supports native paths', null, ErrorDestination.Telemetry);
                }
                else {
                    var initializeResponse = response;
                    initializeResponse.body = {};
                    this.initializeRequest(initializeResponse, args);
                }
            }
            else if (request.command === 'launch') {
                this.launchRequest(response, request.arguments);
            }
            else if (request.command === 'attach') {
                this.attachRequest(response, request.arguments);
            }
            else if (request.command === 'disconnect') {
                this.disconnectRequest(response, request.arguments);
            }
            else if (request.command === 'setBreakpoints') {
                this.setBreakPointsRequest(response, request.arguments);
            }
            else if (request.command === 'setFunctionBreakpoints') {
                this.setFunctionBreakPointsRequest(response, request.arguments);
            }
            else if (request.command === 'setExceptionBreakpoints') {
                this.setExceptionBreakPointsRequest(response, request.arguments);
            }
            else if (request.command === 'configurationDone') {
                this.configurationDoneRequest(response, request.arguments);
            }
            else if (request.command === 'continue') {
                this.continueRequest(response, request.arguments);
            }
            else if (request.command === 'next') {
                this.nextRequest(response, request.arguments);
            }
            else if (request.command === 'stepIn') {
                this.stepInRequest(response, request.arguments);
            }
            else if (request.command === 'stepOut') {
                this.stepOutRequest(response, request.arguments);
            }
            else if (request.command === 'stepBack') {
                this.stepBackRequest(response, request.arguments);
            }
            else if (request.command === 'restartFrame') {
                this.restartFrameRequest(response, request.arguments);
            }
            else if (request.command === 'goto') {
                this.gotoRequest(response, request.arguments);
            }
            else if (request.command === 'pause') {
                this.pauseRequest(response, request.arguments);
            }
            else if (request.command === 'stackTrace') {
                this.stackTraceRequest(response, request.arguments);
            }
            else if (request.command === 'scopes') {
                this.scopesRequest(response, request.arguments);
            }
            else if (request.command === 'variables') {
                this.variablesRequest(response, request.arguments);
            }
            else if (request.command === 'setVariable') {
                this.setVariableRequest(response, request.arguments);
            }
            else if (request.command === 'source') {
                this.sourceRequest(response, request.arguments);
            }
            else if (request.command === 'threads') {
                this.threadsRequest(response);
            }
            else if (request.command === 'evaluate') {
                this.evaluateRequest(response, request.arguments);
            }
            else if (request.command === 'stepInTargets') {
                this.stepInTargetsRequest(response, request.arguments);
            }
            else if (request.command === 'gotoTargets') {
                this.gotoTargetsRequest(response, request.arguments);
            }
            else if (request.command === 'completions') {
                this.completionsRequest(response, request.arguments);
            }
            else {
                this.customRequest(request.command, response, request.arguments);
            }
        }
        catch (e) {
            this.sendErrorResponse(response, 1104, '{_stack}', { _exception: e.message, _stack: e.stack }, ErrorDestination.Telemetry);
        }
    };
    DebugSession.prototype.initializeRequest = function (response, args) {
        // This default debug adapter does not support conditional breakpoints.
        response.body.supportsConditionalBreakpoints = false;
        // This default debug adapter does not support hit conditional breakpoints.
        response.body.supportsHitConditionalBreakpoints = false;
        // This default debug adapter does not support function breakpoints.
        response.body.supportsFunctionBreakpoints = false;
        // This default debug adapter implements the 'configurationDone' request.
        response.body.supportsConfigurationDoneRequest = true;
        // This default debug adapter does not support hovers based on the 'evaluate' request.
        response.body.supportsEvaluateForHovers = false;
        // This default debug adapter does not support the 'stepBack' request.
        response.body.supportsStepBack = false;
        // This default debug adapter does not support the 'setVariable' request.
        response.body.supportsSetVariable = false;
        // This default debug adapter does not support the 'restartFrame' request.
        response.body.supportsRestartFrame = false;
        // This default debug adapter does not support the 'stepInTargetsRequest' request.
        response.body.supportsStepInTargetsRequest = false;
        // This default debug adapter does not support the 'gotoTargetsRequest' request.
        response.body.supportsGotoTargetsRequest = false;
        // This default debug adapter does not support the 'completionsRequest' request.
        response.body.supportsCompletionsRequest = false;
        this.sendResponse(response);
    };
    DebugSession.prototype.disconnectRequest = function (response, args) {
        this.sendResponse(response);
        this.shutdown();
    };
    DebugSession.prototype.launchRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.attachRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.setBreakPointsRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.setFunctionBreakPointsRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.setExceptionBreakPointsRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.configurationDoneRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.continueRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.nextRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.stepInRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.stepOutRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.stepBackRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.restartFrameRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.gotoRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.pauseRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.sourceRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.threadsRequest = function (response) {
        this.sendResponse(response);
    };
    DebugSession.prototype.stackTraceRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.scopesRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.variablesRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.setVariableRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.evaluateRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.stepInTargetsRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.gotoTargetsRequest = function (response, args) {
        this.sendResponse(response);
    };
    DebugSession.prototype.completionsRequest = function (response, args) {
        this.sendResponse(response);
    };
    /**
     * Override this hook to implement custom requests.
     */
    DebugSession.prototype.customRequest = function (command, response, args) {
        this.sendErrorResponse(response, 1014, 'unrecognized request', null, ErrorDestination.Telemetry);
    };
    //---- protected -------------------------------------------------------------------------------------------------
    DebugSession.prototype.convertClientLineToDebugger = function (line) {
        if (this._debuggerLinesStartAt1) {
            return this._clientLinesStartAt1 ? line : line + 1;
        }
        return this._clientLinesStartAt1 ? line - 1 : line;
    };
    DebugSession.prototype.convertDebuggerLineToClient = function (line) {
        if (this._debuggerLinesStartAt1) {
            return this._clientLinesStartAt1 ? line : line - 1;
        }
        return this._clientLinesStartAt1 ? line + 1 : line;
    };
    DebugSession.prototype.convertClientColumnToDebugger = function (column) {
        if (this._debuggerColumnsStartAt1) {
            return this._clientColumnsStartAt1 ? column : column + 1;
        }
        return this._clientColumnsStartAt1 ? column - 1 : column;
    };
    DebugSession.prototype.convertDebuggerColumnToClient = function (column) {
        if (this._debuggerColumnsStartAt1) {
            return this._clientColumnsStartAt1 ? column : column - 1;
        }
        return this._clientColumnsStartAt1 ? column + 1 : column;
    };
    DebugSession.prototype.convertClientPathToDebugger = function (clientPath) {
        if (this._clientPathsAreURIs != this._debuggerPathsAreURIs) {
            if (this._clientPathsAreURIs) {
                return DebugSession.uri2path(clientPath);
            }
            else {
                return DebugSession.path2uri(clientPath);
            }
        }
        return clientPath;
    };
    DebugSession.prototype.convertDebuggerPathToClient = function (debuggerPath) {
        if (this._debuggerPathsAreURIs != this._clientPathsAreURIs) {
            if (this._debuggerPathsAreURIs) {
                return DebugSession.uri2path(debuggerPath);
            }
            else {
                return DebugSession.path2uri(debuggerPath);
            }
        }
        return debuggerPath;
    };
    //---- private -------------------------------------------------------------------------------
    DebugSession.path2uri = function (str) {
        var pathName = str.replace(/\\/g, '/');
        if (pathName[0] !== '/') {
            pathName = '/' + pathName;
        }
        return encodeURI('file://' + pathName);
    };
    DebugSession.uri2path = function (url) {
        return Url.parse(url).pathname;
    };
    /*
    * If argument starts with '_' it is OK to send its value to telemetry.
    */
    DebugSession.formatPII = function (format, excludePII, args) {
        return format.replace(DebugSession._formatPIIRegexp, function (match, paramName) {
            if (excludePII && paramName.length > 0 && paramName[0] !== '_') {
                return match;
            }
            return args[paramName] && args.hasOwnProperty(paramName) ?
                args[paramName] :
                match;
        });
    };
    DebugSession._formatPIIRegexp = /{([^}]+)}/g;
    return DebugSession;
}(protocol_1.ProtocolServer));
exports.DebugSession = DebugSession;
//# sourceMappingURL=data:application/json;base64,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