/**
 * A language server message
 */
export interface Message {
    jsonrpc: string;
}
/**
 * Request message
 */
export interface RequestMessage extends Message {
    /**
     * The request id.
     */
    id: number | string;
    /**
     * The method to be invoked.
     */
    method: string;
    /**
     * The method's params.
     */
    params?: any;
}
/**
 * Predefined error codes.
 */
export declare namespace ErrorCodes {
    const ParseError: number;
    const InvalidRequest: number;
    const MethodNotFound: number;
    const InvalidParams: number;
    const InternalError: number;
    const serverErrorStart: number;
    const serverErrorEnd: number;
    const MessageWriteError: number;
    const MessageReadError: number;
}
export interface ResponseErrorLiteral<D> {
    /**
     * A number indicating the error type that occured.
     */
    code: number;
    /**
     * A string providing a short decription of the error.
     */
    message: string;
    /**
     * A Primitive or Structured value that contains additional
     * information about the error. Can be omitted.
     */
    data?: D;
}
/**
 * A error object return in a response in case a request
 * has failed.
 */
export declare class ResponseError<D> extends Error {
    code: number;
    message: string;
    data: D;
    constructor(code: number, message: string, data?: D);
    toJson(): ResponseErrorLiteral<D>;
}
/**
 * A response message.
 */
export interface ResponseMessage extends Message {
    /**
     * The request id.
     */
    id: number | string;
    /**
     * The result of a request. This can be omitted in
     * the case of an error.
     */
    result?: any;
    /**
     * The error object in case a request fails.
     */
    error?: ResponseErrorLiteral<any>;
}
/**
 * A interface to type the request parameter / response pair
 */
export interface RequestType<P, R, E> {
    method: string;
}
/**
 * Notification Message
 */
export interface NotificationMessage extends Message {
    /**
     * The method to be invoked.
     */
    method: string;
    /**
     * The notification's params.
     */
    params?: any;
}
export interface NotificationType<P> {
    method: string;
}
/**
 * Tests if the given message is a request message
 */
export declare function isRequestMessage(message: Message): message is RequestMessage;
/**
 * Tests if the given message is a notification message
 */
export declare function isNotificationMessage(message: Message): message is NotificationMessage;
/**
 * Tests if the given message is a response message
 */
export declare function isReponseMessage(message: Message): message is ResponseMessage;
