/* --------------------------------------------------------------------------------------------
 * Copyright (c) Microsoft Corporation. All rights reserved.
 * Licensed under the MIT License. See License.txt in the project root for license information.
 * ------------------------------------------------------------------------------------------ */
'use strict';
var code = require('vscode');
var proto = require('./protocol');
var is = require('./utils/is');
var protocolCompletionItem_1 = require('./protocolCompletionItem');
var protocolCodeLens_1 = require('./protocolCodeLens');
function asTextDocumentIdentifier(textDocument) {
    return {
        uri: textDocument.uri.toString()
    };
}
exports.asTextDocumentIdentifier = asTextDocumentIdentifier;
function asOpenTextDocumentParams(textDocument) {
    return {
        textDocument: {
            uri: textDocument.uri.toString(),
            languageId: textDocument.languageId,
            version: textDocument.version,
            text: textDocument.getText()
        }
    };
}
exports.asOpenTextDocumentParams = asOpenTextDocumentParams;
function isTextDocumentChangeEvent(value) {
    var candidate = value;
    return is.defined(candidate.document) && is.defined(candidate.contentChanges);
}
function isTextDocument(value) {
    var candidate = value;
    return is.defined(candidate.uri) && is.defined(candidate.version);
}
function asChangeTextDocumentParams(arg) {
    if (isTextDocument(arg)) {
        var result = {
            textDocument: {
                uri: arg.uri.toString(),
                version: arg.version
            },
            contentChanges: [{ text: arg.getText() }]
        };
        return result;
    }
    else if (isTextDocumentChangeEvent(arg)) {
        var document_1 = arg.document;
        var result = {
            textDocument: {
                uri: document_1.uri.toString(),
                version: document_1.version
            },
            contentChanges: arg.contentChanges.map(function (change) {
                var range = change.range;
                return {
                    range: {
                        start: { line: range.start.line, character: range.start.character },
                        end: { line: range.end.line, character: range.end.character }
                    },
                    rangeLength: change.rangeLength,
                    text: change.text
                };
            })
        };
        return result;
    }
    else {
        throw Error('Unsupported text document change parameter');
    }
}
exports.asChangeTextDocumentParams = asChangeTextDocumentParams;
function asCloseTextDocumentParams(textDocument) {
    return {
        textDocument: asTextDocumentIdentifier(textDocument)
    };
}
exports.asCloseTextDocumentParams = asCloseTextDocumentParams;
function asSaveTextDocumentParams(textDocument) {
    return {
        textDocument: asTextDocumentIdentifier(textDocument)
    };
}
exports.asSaveTextDocumentParams = asSaveTextDocumentParams;
function asTextDocumentPositionParams(textDocument, position) {
    return {
        textDocument: asTextDocumentIdentifier(textDocument),
        position: asWorkerPosition(position)
    };
}
exports.asTextDocumentPositionParams = asTextDocumentPositionParams;
function asWorkerPosition(position) {
    return { line: position.line, character: position.character };
}
exports.asWorkerPosition = asWorkerPosition;
function asRange(value) {
    if (is.undefined(value)) {
        return undefined;
    }
    else if (is.nil(value)) {
        return null;
    }
    return { start: asPosition(value.start), end: asPosition(value.end) };
}
exports.asRange = asRange;
function asPosition(value) {
    if (is.undefined(value)) {
        return undefined;
    }
    else if (is.nil(value)) {
        return null;
    }
    return { line: value.line, character: value.character };
}
exports.asPosition = asPosition;
function set(value, func) {
    if (is.defined(value)) {
        func();
    }
}
function asDiagnosticSeverity(value) {
    switch (value) {
        case code.DiagnosticSeverity.Error:
            return proto.DiagnosticSeverity.Error;
        case code.DiagnosticSeverity.Warning:
            return proto.DiagnosticSeverity.Warning;
        case code.DiagnosticSeverity.Information:
            return proto.DiagnosticSeverity.Information;
        case code.DiagnosticSeverity.Hint:
            return proto.DiagnosticSeverity.Hint;
    }
}
exports.asDiagnosticSeverity = asDiagnosticSeverity;
function asDiagnostic(item) {
    var result = proto.Diagnostic.create(asRange(item.range), item.message);
    set(item.severity, function () { return result.severity = asDiagnosticSeverity(item.severity); });
    set(item.code, function () { return result.code = item.code; });
    set(item.source, function () { return result.source = item.source; });
    return result;
}
exports.asDiagnostic = asDiagnostic;
function asDiagnostics(items) {
    if (is.undefined(items) || is.nil(items)) {
        return items;
    }
    return items.map(asDiagnostic);
}
exports.asDiagnostics = asDiagnostics;
function asCompletionItem(item) {
    var result = { label: item.label };
    set(item.detail, function () { return result.detail = item.detail; });
    set(item.documentation, function () { return result.documentation = item.documentation; });
    set(item.filterText, function () { return result.filterText = item.filterText; });
    set(item.insertText, function () { return result.insertText = item.insertText; });
    // Protocol item kind is 1 based, codes item kind is zero based.
    set(item.kind, function () { return result.kind = item.kind + 1; });
    set(item.sortText, function () { return result.sortText = item.sortText; });
    set(item.textEdit, function () { return result.textEdit = asTextEdit(item.textEdit); });
    if (item instanceof protocolCompletionItem_1.default) {
        set(item.data, function () { return result.data = item.data; });
    }
    return result;
}
exports.asCompletionItem = asCompletionItem;
function asTextEdit(edit) {
    return { range: asRange(edit.range), newText: edit.newText };
}
exports.asTextEdit = asTextEdit;
function asReferenceParams(textDocument, position, options) {
    return {
        textDocument: asTextDocumentIdentifier(textDocument),
        position: asWorkerPosition(position),
        context: { includeDeclaration: options.includeDeclaration }
    };
}
exports.asReferenceParams = asReferenceParams;
function asCodeActionContext(context) {
    if (is.undefined(context) || is.nil(context)) {
        return context;
    }
    return proto.CodeActionContext.create(asDiagnostics(context.diagnostics));
}
exports.asCodeActionContext = asCodeActionContext;
function asCommand(item) {
    var result = proto.Command.create(item.title, item.command);
    if (is.defined(item.arguments))
        result.arguments = item.arguments;
    return result;
}
exports.asCommand = asCommand;
function asCodeLens(item) {
    var result = proto.CodeLens.create(asRange(item.range));
    if (is.defined(item.command))
        result.command = asCommand(item.command);
    if (item instanceof protocolCodeLens_1.default) {
        if (is.defined(item.data))
            result.data = item.data;
    }
    return result;
}
exports.asCodeLens = asCodeLens;
function asFormattingOptions(item) {
    return { tabSize: item.tabSize, insertSpaces: item.insertSpaces };
}
exports.asFormattingOptions = asFormattingOptions;
function asDocumentSymbolParams(textDocument) {
    return {
        textDocument: asTextDocumentIdentifier(textDocument)
    };
}
exports.asDocumentSymbolParams = asDocumentSymbolParams;
function asCodeLensParams(textDocument) {
    return {
        textDocument: asTextDocumentIdentifier(textDocument)
    };
}
exports.asCodeLensParams = asCodeLensParams;
