///<reference path="..\Declarations\node\node.d.ts" />
var Config = require("./Config");
var Context = require("./Context");
var ExceptionTracking = require("../AutoCollection/Exceptions");
var ContractsModule = require("../Library/Contracts");
var Channel = require("./Channel");
var RequestTracking = require("../AutoCollection/Requests");
var Sender = require("./Sender");
var Util = require("./Util");
var Client = (function () {
    /**
     * Constructs a new client of the client
     * @param iKey the instrumentation key to use (read from environment variable if not specified)
     */
    function Client(iKey) {
        var config = new Config(iKey);
        this.config = config;
        this.context = new Context();
        this.commonProperties = {};
        var sender = new Sender(function () { return config.endpointUrl; });
        this.channel = new Channel(function () { return config.disableAppInsights; }, function () { return config.maxBatchSize; }, function () { return config.maxBatchIntervalMs; }, sender);
    }
    /**
     * Log a user action or other occurrence.
     * @param   name    A string to identify this event in the portal.
     * @param   properties  map[string, string] - additional data used to filter events and metrics in the portal. Defaults to empty.
     * @param   measurements    map[string, number] - metrics associated with this event, displayed in Metrics Explorer on the portal. Defaults to empty.
     */
    Client.prototype.trackEvent = function (name, properties, measurements) {
        var event = new ContractsModule.Contracts.EventData();
        event.name = name;
        event.properties = properties;
        event.measurements = measurements;
        var data = new ContractsModule.Contracts.Data();
        data.baseType = "EventData";
        data.baseData = event;
        this.track(data);
    };
    /**
     * Log a trace message
     * @param   message    A string to identify this event in the portal.
     * @param   properties  map[string, string] - additional data used to filter events and metrics in the portal. Defaults to empty.
     */
    Client.prototype.trackTrace = function (message, severityLevel, properties) {
        var trace = new ContractsModule.Contracts.MessageData();
        trace.message = message;
        trace.properties = properties;
        if (!isNaN(severityLevel)) {
            trace.severityLevel = severityLevel;
        }
        else {
            trace.severityLevel = ContractsModule.Contracts.SeverityLevel.Information;
        }
        var data = new ContractsModule.Contracts.Data();
        data.baseType = "MessageData";
        data.baseData = trace;
        this.track(data);
    };
    /**
     * Log an exception you have caught.
     * @param   exception   An Error from a catch clause, or the string error message.
     * @param   properties  map[string, string] - additional data used to filter events and metrics in the portal. Defaults to empty.
     * @param   measurements    map[string, number] - metrics associated with this event, displayed in Metrics Explorer on the portal. Defaults to empty.
     */
    Client.prototype.trackException = function (exception, properties) {
        if (!Util.isError(exception)) {
            exception = new Error(exception);
        }
        var data = ExceptionTracking.getExceptionData(exception, true, properties);
        this.track(data);
    };
    /**
     * * Log a numeric value that is not associated with a specific event. Typically used to send regular reports of performance indicators.
     * To send a single measurement, use just the first two parameters. If you take measurements very frequently, you can reduce the
     * telemetry bandwidth by aggregating multiple measurements and sending the resulting average at intervals.
     *
     * @param name   A string that identifies the metric.
     * @param value  The value of the metric
     * @param count  the number of samples used to get this value
     * @param min    the min sample for this set
     * @param max    the max sample for this set
     * @param stdDev the standard deviation of the set
     */
    Client.prototype.trackMetric = function (name, value, count, min, max, stdDev) {
        var metrics = new ContractsModule.Contracts.MetricData(); // todo: enable client-batching of these
        metrics.metrics = [];
        var metric = new ContractsModule.Contracts.DataPoint();
        metric.count = !isNaN(count) ? count : 1;
        metric.kind = ContractsModule.Contracts.DataPointType.Aggregation;
        metric.max = !isNaN(max) ? max : value;
        metric.min = !isNaN(min) ? min : value;
        metric.name = name;
        metric.stdDev = !isNaN(stdDev) ? stdDev : 0;
        metric.value = value;
        metrics.metrics.push(metric);
        var data = new ContractsModule.Contracts.Data();
        data.baseType = "MetricData";
        data.baseData = metrics;
        this.track(data);
    };
    Client.prototype.trackRequest = function (request, response, properties) {
        RequestTracking.trackRequest(this, request, response, properties);
    };
    /**
     * Immediately send all queued telemetry.
     */
    Client.prototype.sendPendingData = function (callback) {
        this.channel.triggerSend(false, callback);
    };
    Client.prototype.getEnvelope = function (data, tagOverrides) {
        if (data && data.baseData) {
            data.baseData.ver = 2;
            // if no properties are specified just add the common ones
            if (!data.baseData.properties) {
                data.baseData.properties = this.commonProperties;
            }
            else {
                // otherwise, check each of the common ones
                for (var name in this.commonProperties) {
                    // only override if the property `name` has not been set on this item
                    if (!data.baseData.properties[name]) {
                        data.baseData.properties[name] = this.commonProperties[name];
                    }
                }
            }
        }
        // sanitize properties
        data.baseData.properties = Util.validateStringMap(data.baseData.properties);
        var envelope = new ContractsModule.Contracts.Envelope();
        envelope.data = data;
        envelope.appVer = this.context.tags[this.context.keys.applicationVersion];
        envelope.iKey = this.config.instrumentationKey;
        // this is kind of a hack, but the envelope name is always the same as the data name sans the chars "data"
        envelope.name = "Microsoft.ApplicationInsights." + data.baseType.substr(0, data.baseType.length - 4);
        envelope.os = this.context.tags[this.context.keys.deviceOS];
        envelope.osVer = this.context.tags[this.context.keys.deviceOSVersion];
        envelope.seq = (Client._sequenceNumber++).toString();
        envelope.tags = tagOverrides || this.context.tags;
        envelope.time = (new Date()).toISOString();
        envelope.ver = 1;
        return envelope;
    };
    /**
     * Generic track method for all telemetry types
     * @param data the telemetry to send
     * @param tagOverrides the context tags to use for this telemetry which overwrite default context values
     */
    Client.prototype.track = function (data, tagOverrides) {
        var envelope = this.getEnvelope(data, tagOverrides);
        this.channel.send(envelope);
    };
    Client._sequenceNumber = 0;
    return Client;
})();
module.exports = Client;
